<?php
namespace Core\Classes;

class Router {
    private $routes = [];

    public function add($method, $path, $handler) {
        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'handler' => $handler
        ];
    }

    public function dispatch($method, $uri) {
        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }

            $pattern = $this->convertToRegex($route['path']);
            
            if (preg_match($pattern, $uri, $matches)) {
                array_shift($matches);
                
                if (is_callable($route['handler'])) {
                    return call_user_func_array($route['handler'], $matches);
                }
                
                if (is_string($route['handler'])) {
                    return $this->callControllerMethod($route['handler'], $matches);
                }
            }
        }
        
        http_response_code(404);
        echo "404 - Página no encontrada: " . htmlspecialchars($uri);
    }

    private function convertToRegex($path) {
        $path = preg_replace('/\{([a-zA-Z0-9_]+)\}/', '([a-zA-Z0-9_]+)', $path);
        return '#^' . $path . '$#';
    }

    private function callControllerMethod($handler, $params) {
        list($controllerName, $method) = explode('@', $handler);
        
        $controllerClass = "Core\\Controllers\\{$controllerName}";
        
        if (!class_exists($controllerClass)) {
            die("Error: Controlador no encontrado - {$controllerClass}");
        }
        
        $controller = new $controllerClass();
        
        if (!method_exists($controller, $method)) {
            die("Error: Método no encontrado - {$controllerClass}::{$method}");
        }
        
        return call_user_func_array([$controller, $method], $params);
    }
}
