<?php
namespace Core\Models;

use Core\Classes\Database;

class Usuario {
    private $db;
    private $table = 'usuarios';

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    /**
     * Obtiene todos los usuarios
     */
    public function getAll() {
        $stmt = $this->db->query("SELECT * FROM {$this->table} WHERE deleted_at IS NULL ORDER BY created_at DESC");
        return $stmt->fetchAll();
    }

    /**
     * Obtiene un usuario por ID
     */
    public function getById($id) {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE id = ? AND deleted_at IS NULL");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }

    /**
     * Obtiene un usuario por email
     */
    public function getByEmail($email) {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE email = ? AND deleted_at IS NULL AND estado = 'activo'");
        $stmt->execute([$email]);
        return $stmt->fetch();
    }

    /**
     * Obtiene usuarios de una empresa
     */
    public function getByEmpresa($empresa_id) {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE empresa_id = ? AND deleted_at IS NULL ORDER BY created_at DESC");
        $stmt->execute([$empresa_id]);
        return $stmt->fetchAll();
    }

    /**
     * Crea un nuevo usuario
     */
    public function create($data) {
        $sql = "INSERT INTO {$this->table} 
                (empresa_id, rut, nombres, apellidos, email, password_hash, telefono, rol, estado, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
        
        $stmt = $this->db->prepare($sql);
        
        // Hash de la contraseña
        $password_hash = password_hash($data['password'], PASSWORD_DEFAULT);
        
        return $stmt->execute([
            $data['empresa_id'],
            $data['rut'],
            $data['nombres'],
            $data['apellidos'],
            $data['email'],
            $password_hash,
            $data['telefono'],
            $data['rol'],
            $data['estado']
        ]);
    }

    /**
     * Actualiza un usuario
     */
    public function update($id, $data) {
        // Si hay contraseña, actualizarla
        if (isset($data['password']) && !empty($data['password'])) {
            $sql = "UPDATE {$this->table} 
                    SET rut = ?, nombres = ?, apellidos = ?, email = ?, password_hash = ?, 
                        telefono = ?, rol = ?, estado = ?, updated_at = NOW() 
                    WHERE id = ?";
            
            $password_hash = password_hash($data['password'], PASSWORD_DEFAULT);
            
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([
                $data['rut'],
                $data['nombres'],
                $data['apellidos'],
                $data['email'],
                $password_hash,
                $data['telefono'],
                $data['rol'],
                $data['estado'],
                $id
            ]);
        } else {
            // Sin actualizar contraseña
            $sql = "UPDATE {$this->table} 
                    SET rut = ?, nombres = ?, apellidos = ?, email = ?, 
                        telefono = ?, rol = ?, estado = ?, updated_at = NOW() 
                    WHERE id = ?";
            
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([
                $data['rut'],
                $data['nombres'],
                $data['apellidos'],
                $data['email'],
                $data['telefono'],
                $data['rol'],
                $data['estado'],
                $id
            ]);
        }
    }

    /**
     * Elimina un usuario (soft delete)
     */
    public function delete($id) {
        $stmt = $this->db->prepare("UPDATE {$this->table} SET deleted_at = NOW() WHERE id = ?");
        return $stmt->execute([$id]);
    }

    /**
     * Verifica si un email ya existe
     */
    public function emailExists($email, $excludeId = null) {
        if ($excludeId) {
            $stmt = $this->db->prepare("SELECT COUNT(*) FROM {$this->table} WHERE email = ? AND id != ? AND deleted_at IS NULL");
            $stmt->execute([$email, $excludeId]);
        } else {
            $stmt = $this->db->prepare("SELECT COUNT(*) FROM {$this->table} WHERE email = ? AND deleted_at IS NULL");
            $stmt->execute([$email]);
        }
        return $stmt->fetchColumn() > 0;
    }
}
