<?php
session_start();
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once 'config.php';

function e($text) { return htmlspecialchars($text ?? '', ENT_QUOTES, 'UTF-8'); }
function user() { return $_SESSION['user'] ?? null; }

$user = user();
// Solo administradores (Rol 1 o 2) pueden crear/editar rendiciones
if (!$user || $user['rol'] > 2) { 
    header('Location: index.php');
    exit;
}

$config = include 'config.php';
$dsn = "mysql:host={$config['db']['host']};dbname={$config['db']['dbname']};charset=utf8mb4";
$pdo = new PDO($dsn, $config['db']['user'], $config['db']['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
]);

$rendicion = [
    'id' => 0,
    'ssr_id' => 0,
    'titulo' => '',
    'fecha' => date('Y-m-d'),
    'contenido' => '',
    'estado' => 'borrador'
];
$msg = $error = null;
$es_edicion = false;
$accion = 'Crear';

$rendicion_id = (int)($_GET['id'] ?? 0);

// 1. Lógica de Edición: Cargar datos si existe ID
if ($rendicion_id) {
    $stmt = $pdo->prepare("SELECT * FROM rendiciones WHERE id = ?");
    $stmt->execute([$rendicion_id]);
    $data = $stmt->fetch();
    
    if ($data) {
        $rendicion = $data;
        $es_edicion = true;
        $accion = 'Editar';
    } else {
        $error = "❌ Rendición ID {$rendicion_id} no encontrada.";
        $rendicion_id = 0; // Forzar modo creación si falla la carga
    }
}

// Obtener SSRs para el selector (solo si no es Edición o si el usuario es SuperAdmin)
if ($user['rol'] == 1) {
    $stmt = $pdo->query("SELECT id, nombre FROM ssr ORDER BY nombre");
    $ssrs = $stmt->fetchAll();
} else {
    // Si es Admin SSR, solo puede ver y seleccionar su propio SSR
    $ssrs = [['id' => $user['ssr_id'], 'nombre' => $user['ssr_nombre']]];
}


// 2. Lógica de Manejo POST (Guardar/Actualizar)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && !$error) {
    $titulo = trim($_POST['titulo']);
    $ssr_id_post = (int)($_POST['ssr_id'] ?? $user['ssr_id']);
    $fecha = $_POST['fecha'];
    $contenido = trim($_POST['contenido']);
    $estado = $_POST['estado'] ?? 'borrador';

    // Validaciones
    if (empty($titulo) || empty($fecha) || empty($contenido) || !$ssr_id_post) {
        $error = "❌ Todos los campos son obligatorios.";
    } elseif ($user['rol'] == 2 && $ssr_id_post !== $user['ssr_id']) {
        $error = "❌ No tienes permiso para asignar rendiciones a otro SSR.";
    } else {
        // Asignar los valores a la variable $rendicion para que persistan en el formulario
        $rendicion['ssr_id'] = $ssr_id_post;
        $rendicion['titulo'] = $titulo;
        $rendicion['fecha'] = $fecha;
        $rendicion['contenido'] = $contenido;
        $rendicion['estado'] = $estado;

        if ($es_edicion) {
            // UPDATE: Actualizar rendición existente
            $stmt = $pdo->prepare("UPDATE rendiciones SET ssr_id = ?, titulo = ?, fecha = ?, contenido = ?, estado = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$ssr_id_post, $titulo, $fecha, $contenido, $estado, $rendicion_id]);
            $msg = "✅ Rendición actualizada con éxito.";
        } else {
            // INSERT: Crear nueva rendición
            $stmt = $pdo->prepare("INSERT INTO rendiciones (ssr_id, titulo, fecha, contenido, estado, created_by) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$ssr_id_post, $titulo, $fecha, $contenido, $estado, $user['id']]);
            $rendicion_id = $pdo->lastInsertId();
            $es_edicion = true;
            $accion = 'Editar';
            $msg = "✅ Rendición creada con éxito. Puede subir PDFs ahora.";
            
            // Redirigir a la edición para poder seguir trabajando y subir PDFs
            header("Location: index.php?p=nueva_rendicion&id={$rendicion_id}&msg=" . urlencode($msg));
            exit;
        }
    }
}

$titulo_pagina = ($es_edicion) ? "Editar Rendición #{$rendicion_id}" : "Nueva Rendición";
$volver_url = ($es_edicion) ? 'rendiciones_admin.php?ssr=' . $rendicion['ssr_id'] : 'index.php?p=dashboard';

// Si hay un mensaje en la URL después de la redirección de INSERT
if (isset($_GET['msg'])) {
    $msg = htmlspecialchars($_GET['msg']);
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>📝 <?= $accion ?> Rendición</title>
    <meta charset="utf-8">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root { --primary: #77216F; --secondary: #5E2750; --success: #28a745; --danger: #dc3545; }
        *{margin:0;padding:0;box-sizing:border-box;}
        body{font-family:'Segoe UI',Arial,sans-serif;background:#f8f9fa;padding:20px;min-height:100vh;}
        .container{max-width:900px;margin:0 auto;background:white;padding:30px;border-radius:12px;box-shadow:0 0 15px rgba(0,0,0,0.1);}
        .header{border-bottom:2px solid var(--primary);padding-bottom:15px;margin-bottom:30px;text-align:center;}
        .header h1{color:var(--primary);}
        .form-group{margin-bottom:15px;}
        .form-group label{display:block;margin-bottom:5px;font-weight:600;color:#555;}
        .form-group input[type="text"], .form-group input[type="date"], .form-group textarea, .form-group select{width:100%;padding:10px;border:1px solid #ddd;border-radius:6px;font-size:16px;}
        .btn{padding:10px 20px;border:none;border-radius:6px;font-weight:600;cursor:pointer;text-decoration:none;transition:background 0.3s;}
        .btn-primary{background:var(--primary);color:white;}
        .btn-primary:hover{background:var(--secondary);}
        .msg-success{background:#d4edda;color:#155724;padding:15px;border-radius:8px;margin-bottom:20px;}
        .msg-error{background:#f8d7da;color:#721c24;padding:15px;border-radius:8px;margin-bottom:20px;}
        .pdf-section{margin-top:30px;border-top:1px solid #eee;padding-top:20px;}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-file-invoice"></i> <?= e($titulo_pagina) ?></h1>
            <a href="<?= e($volver_url) ?>" class="btn" style="background:#6c757d;color:white;">← Volver</a>
        </div>
        
        <?php if($msg): ?><div class="msg-success"><?= $msg ?></div><?php endif; ?>
        <?php if($error): ?><div class="msg-error"><?= $error ?></div><?php endif; ?>

        <form method="POST">
            <?php if($user['rol'] == 1): ?>
            <div class="form-group">
                <label for="ssr_id">SSR Asignado *</label>
                <select id="ssr_id" name="ssr_id" required>
                    <?php foreach($ssrs as $ssr): ?>
                    <option value="<?= e($ssr['id']) ?>" <?= $rendicion['ssr_id'] == $ssr['id'] ? 'selected' : '' ?>>
                        <?= e($ssr['nombre']) ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php else: ?>
            <input type="hidden" name="ssr_id" value="<?= e($user['ssr_id']) ?>">
            <p style="margin-bottom:15px;color:#555;">**Asignado a: <?= e($user['ssr_nombre']) ?>**</p>
            <?php endif; ?>

            <div class="form-group">
                <label for="titulo">Título de la Rendición *</label>
                <input type="text" id="titulo" name="titulo" value="<?= e($rendicion['titulo']) ?>" required maxlength="255">
            </div>

            <div class="form-group">
                <label for="fecha">Fecha de Rendición *</label>
                <input type="date" id="fecha" name="fecha" value="<?= e($rendicion['fecha']) ?>" required>
            </div>
            
            <div class="form-group">
                <label for="contenido">Contenido/Resumen *</label>
                <textarea id="contenido" name="contenido" rows="5" required><?= e($rendicion['contenido']) ?></textarea>
            </div>

            <div class="form-group">
                <label for="estado">Estado</label>
                <select id="estado" name="estado">
                    <option value="borrador" <?= $rendicion['estado'] == 'borrador' ? 'selected' : '' ?>>Borrador</option>
                    <option value="publicada" <?= $rendicion['estado'] == 'publicada' ? 'selected' : '' ?>>Publicada</option>
                </select>
            </div>

            <div style="margin-top:20px;">
                <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> <?= $accion ?> Rendición</button>
            </div>
        </form>

        <?php if($es_edicion): ?>
        <div class="pdf-section">
            <h3><i class="fas fa-file-pdf"></i> Archivos PDF (Anexos)</h3>
            <p style="color:#666;margin-bottom:15px;">Use el botón "Subir PDF" para adjuntar documentos a esta rendición.</p>
            <a href="index.php?p=subir_pdf&id=<?= $rendicion_id ?>" class="btn" style="background:var(--success);color:white;">
                <i class="fas fa-upload"></i> Subir/Reemplazar PDF
            </a>
        </div>
        <?php endif; ?>
    </div>
</body>
</html>