<?php
/**
 * Controlador de Casos para el Portal de Clientes
 *
 * Maneja la creación de nuevos casos por parte de los clientes.
 * Se encarga de:
 * - Validar los datos del formulario.
 * - Procesar la subida segura de archivos adjuntos.
 * - Guardar el nuevo caso en la base de datos.
 * - Enviar notificaciones por correo electrónico al cliente y al administrador.
 */

// Incluir la configuración central. La ruta es relativa y sube varios niveles.
require_once __DIR__ . '/../../../../config/configuracion.php';

// Usar la constante APP_ROOT para cargar los demás archivos.
// Esto asegura que las rutas sean siempre correctas, sin importar cómo se llame al script.
require_once APP_ROOT . '/private/controllers/portal/portal_seguridad.php';
require_once APP_ROOT . '/config/conexion.php'; // Usa el archivo de conexión centralizado.

// Enrutador simple: por ahora, solo maneja la acción 'create'.
$action = $_GET['action'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'create') {
    handle_create_case();
} else {
    // Si se accede al script de forma incorrecta, redirigir al dashboard.
    header('Location: ' . BASE_URL . 'portal/dashboard.php');
    exit;
}

/**
 * Función principal que procesa la creación de un caso.
 */
function handle_create_case() {
    global $conexion;

    // Recoger datos del formulario (POST) y de la sesión.
    $tipificacion = $_POST['tipificacion'] ?? '';
    $descripcion = $_POST['descripcion'] ?? '';
    $cliente_id = $_SESSION['cliente_id'];
    $empresa_id = $_SESSION['cliente_id_empresa'];
    $ruta_adjunto = null; // Inicializar como nulo.

    // Validación básica de campos.
    if (empty($tipificacion) || empty($descripcion)) {
        set_mensaje('Error: El tipo de caso y la descripción son obligatorios.', 'danger');
        header('Location: ' . BASE_URL . 'portal/crear_caso.php');
        exit;
    }

    // --- 1. PROCESAR ARCHIVO ADJUNTO (SI EXISTE) ---
    if (isset($_FILES['adjunto']) && $_FILES['adjunto']['error'] == UPLOAD_ERR_OK) {
        $archivo = $_FILES['adjunto'];
        
        // Crear un nombre de archivo único para evitar sobreescrituras y problemas de seguridad.
        // ej: 1667933512-636a8d10d9b5e-factura.pdf
        $nombre_unico = time() . '-' . uniqid('', true) . '-' . basename($archivo['name']);
        
        // Ruta segura para guardar los archivos, fuera del acceso público.
        $directorio_destino = APP_ROOT . '/private/uploads/';
        $ruta_completa = $directorio_destino . $nombre_unico;

        if (move_uploaded_file($archivo['tmp_name'], $ruta_completa)) {
            $ruta_adjunto = $nombre_unico; // Si el archivo se movió con éxito, guardamos su nombre.
        } else {
            set_mensaje('Error crítico al guardar el archivo adjunto.', 'danger');
            header('Location: ' . BASE_URL . 'portal/crear_caso.php');
            exit;
        }
    }

    // --- 2. INSERTAR EL CASO EN LA BASE DE DATOS ---
    $estado_inicial = 'NUEVO';
    $origen = 'Portal Cliente';
    
    $stmt = $conexion->prepare(
        "INSERT INTO casos (id_cliente, id_empresa, tipificacion, descripcion, estado, origen, fecha_creacion, ruta_adjunto) 
         VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)"
    );
    $stmt->bind_param("issssss", $cliente_id, $empresa_id, $tipificacion, $descripcion, $estado_inicial, $origen, $ruta_adjunto);

    if ($stmt->execute()) {
        $nuevo_caso_id = $stmt->insert_id;
        $stmt->close();
        set_mensaje('Su caso #' . $nuevo_caso_id . ' ha sido creado con éxito.', 'success');

        // --- 3. ENVIAR NOTIFICACIONES POR EMAIL ---
        // Incluimos la librería de email justo donde la necesitamos.
        // La ruta apunta a /private/libs/ donde está el nuevo funciones_email.
        require_once APP_ROOT . '/private/libs/funciones_email.php';

        // Obtener el email del cliente desde la tabla de autenticación.
        $stmt_email = $conexion->prepare("SELECT email FROM clientes_auth WHERE cliente_id = ?");
        $stmt_email->bind_param("i", $cliente_id);
        $stmt_email->execute();
        $resultado_email = $stmt_email->get_result()->fetch_assoc();
        $email_cliente = $resultado_email ? $resultado_email['email'] : null;
        $stmt_email->close();

        // Notificación para el cliente.
        if ($email_cliente) {
            $nombre_cliente = $_SESSION['cliente_nombre'];
            $asunto_cliente = "Confirmación de Caso #" . $nuevo_caso_id;
            $cuerpo_cliente = "<h1>Hola {$nombre_cliente},</h1><p>Hemos recibido tu caso y se ha registrado con el número <strong>#{$nuevo_caso_id}</strong>.</p><p>Puedes ver su estado en cualquier momento desde el portal.</p>";
            enviar_email($email_cliente, $nombre_cliente, $asunto_cliente, $cuerpo_cliente);
        }

        // Notificación para el administrador. (El email debe ser configurable en el futuro).
        $email_admin = 'admin@comite-ssr.cl'; // REEMPLAZAR
        $asunto_admin = "Nuevo Caso Registrado #" . $nuevo_caso_id;
        $cuerpo_admin = "<h1>Nuevo Caso desde el Portal</h1><p>Se ha registrado el caso <strong>#{$nuevo_caso_id}</strong> del cliente {$_SESSION['cliente_nombre']}.</p>";
        enviar_email($email_admin, 'Administrador', $asunto_admin, $cuerpo_admin);

    } else {
        set_mensaje('Error al registrar el caso en la base de datos.', 'danger');
        $stmt->close();
    }
    
    // Redirigir al dashboard para ver el resultado.
    header('Location: ' . BASE_URL . 'portal/dashboard.php');
    exit;
}

/**
 * Función auxiliar para guardar mensajes en la sesión (flash messages).
 */
function set_mensaje($texto, $tipo) {
    $_SESSION['mensaje'] = ['texto' => $texto, 'tipo' => $tipo];
}
?>
