<?php
/**
 * Controlador de Autenticación para el Portal de Clientes (SGReclamos v2.0)
 *
 * Maneja el inicio de sesión (login) y el registro (register) de los clientes.
 * Utiliza la conexión a la base de datos existente via mysqli ($conexion).
 */

// Iniciar o reanudar la sesión para poder gestionar los datos del cliente logueado.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Incluir el archivo de conexión principal de la aplicación.
// La ruta relativa sube 4 niveles desde /private/controllers/portal/ para llegar a la raíz.
require_once __DIR__ . '/../../../../conexion.php'; 

// Asegurar que todas las salidas de este script sean en formato JSON.
header('Content-Type: application/json');

// Determinar la acción solicitada (login o register) desde la URL.
$action = $_GET['action'] ?? '';

// Enrutador: Llama a la función correspondiente según la acción.
if ($action === 'login') {
    handle_login();
} elseif ($action === 'register') {
    handle_register();
} else {
    // Si no se especifica una acción válida, devuelve un error.
    echo json_encode(['status' => 'error', 'message' => 'Acción no válida o no especificada.']);
}

/**
 * Procesa la solicitud de inicio de sesión del cliente.
 */
function handle_login() {
    // Hace que la variable $conexion (de conexion.php) esté disponible dentro de esta función.
    global $conexion; 

    // Solo procesar si la solicitud es de tipo POST.
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['status' => 'error', 'message' => 'Método no permitido.']);
        return;
    }

    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';

    // Validaciones de los datos recibidos.
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['status' => 'error', 'message' => 'Por favor, ingrese un correo electrónico válido.']);
        return;
    }
    if (empty($password)) {
        echo json_encode(['status' => 'error', 'message' => 'Por favor, ingrese su contraseña.']);
        return;
    }

    // Buscar en la tabla de autenticación.
    $stmt_auth = $conexion->prepare("SELECT * FROM clientes_auth WHERE email = ?");
    $stmt_auth->bind_param("s", $email);
    $stmt_auth->execute();
    $result_auth = $stmt_auth->get_result();
    $auth_user = $result_auth->fetch_assoc();
    $stmt_auth->close();

    // Verificar si el usuario existe y si la contraseña es correcta.
    if ($auth_user && password_verify($password, $auth_user['password_hash'])) {
        
        // Si las credenciales son válidas, obtener los datos del cliente.
        $stmt_cliente = $conexion->prepare("SELECT id, nombre, rut, id_empresa FROM clientes WHERE id = ? AND activo = 1");
        $stmt_cliente->bind_param("i", $auth_user['cliente_id']);
        $stmt_cliente->execute();
        $result_cliente = $stmt_cliente->get_result();
        $cliente_data = $result_cliente->fetch_assoc();
        $stmt_cliente->close();

        if (!$cliente_data) {
            echo json_encode(['status' => 'error', 'message' => 'Error: Cliente asociado no encontrado o se encuentra inactivo.']);
            return;
        }
        
        // Regenerar el ID de sesión por seguridad.
        session_regenerate_id(true);

        // Guardar los datos del cliente en la sesión.
        $_SESSION['cliente_id'] = $cliente_data['id'];
        $_SESSION['cliente_nombre'] = $cliente_data['nombre'];
        $_SESSION['cliente_rut'] = $cliente_data['rut'];
        $_SESSION['cliente_id_empresa'] = $cliente_data['id_empresa'];
        $_SESSION['cliente_logged_in'] = true;

        // Enviar respuesta de éxito.
        echo json_encode(['status' => 'success', 'message' => 'Inicio de sesión exitoso.']);

    } else {
        // Credenciales incorrectas.
        echo json_encode(['status' => 'error', 'message' => 'El correo electrónico o la contraseña son incorrectos.']);
    }
}

/**
 * Procesa la solicitud de registro de un nuevo cliente.
 */
function handle_register() {
    global $conexion;

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['status' => 'error', 'message' => 'Método no permitido.']);
        return;
    }

    $rut = $_POST['rut'] ?? '';
    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';
    $password_confirm = $_POST['password_confirm'] ?? '';

    // Validaciones de formulario.
    if (empty($rut) || empty($email) || empty($password)) {
        echo json_encode(['status' => 'error', 'message' => 'Todos los campos son obligatorios.']);
        return;
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['status' => 'error', 'message' => 'El formato del correo electrónico no es válido.']);
        return;
    }
    if ($password !== $password_confirm) {
        echo json_encode(['status' => 'error', 'message' => 'Las contraseñas no coinciden.']);
        return;
    }
    if (strlen($password) < 8) {
        echo json_encode(['status' => 'error', 'message' => 'La contraseña debe tener al menos 8 caracteres.']);
        return;
    }

    try {
        // Verificar si el RUT existe en la tabla `clientes`.
        $stmt_cliente = $conexion->prepare("SELECT id FROM clientes WHERE rut = ?");
        $stmt_cliente->bind_param("s", $rut);
        $stmt_cliente->execute();
        $result_cliente = $stmt_cliente->get_result();
        $cliente = $result_cliente->fetch_assoc();
        $stmt_cliente->close();

        if (!$cliente) {
            echo json_encode(['status' => 'error', 'message' => 'RUT no encontrado. Asegúrese de que esté escrito correctamente o contacte a soporte.']);
            return;
        }
        $cliente_id = $cliente['id'];

        // Verificar que este cliente no tenga ya una cuenta o que el email no esté ya en uso.
        $stmt_check = $conexion->prepare("SELECT id FROM clientes_auth WHERE cliente_id = ? OR email = ?");
        $stmt_check->bind_param("is", $cliente_id, $email);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        
        if ($result_check->num_rows > 0) {
            echo json_encode(['status' => 'error', 'message' => 'Ya existe una cuenta para este cliente o el correo electrónico ya está en uso.']);
            $stmt_check->close();
            return;
        }
        $stmt_check->close();

        // Hashear la contraseña para almacenarla de forma segura.
        $password_hash = password_hash($password, PASSWORD_DEFAULT);

        // Insertar el nuevo registro de autenticación.
        $stmt_insert = $conexion->prepare("INSERT INTO clientes_auth (cliente_id, email, password_hash) VALUES (?, ?, ?)");
        $stmt_insert->bind_param("iss", $cliente_id, $email, $password_hash);
        
        if ($stmt_insert->execute()) {
            echo json_encode(['status' => 'success', 'message' => '¡Cuenta creada con éxito! Ahora puede iniciar sesión.']);
        } else {
            throw new Exception("No se pudo crear la cuenta en la base de datos.");
        }
        $stmt_insert->close();

    } catch (Exception $e) {
        // En un entorno de producción, es mejor registrar este error que mostrarlo.
        error_log('Error en registro: ' . $e->getMessage());
        echo json_encode(['status' => 'error', 'message' => 'Ocurrió un error en el servidor. Por favor, intente más tarde.']);
    }
}
?>
