<?php
/**
 * Editar Usuario Interno
 * Versión: 2.0 Definitiva
 */

if (!defined('COMITEH_APP')) {
    die('Acceso no autorizado');
}

if (!verificar_sesion()) {
    redireccionar('/login');
}

$usuario_rol_actual = $_SESSION['usuario_rol'];
$empresa_id_actual = $_SESSION['empresa_id'];

if (!in_array($usuario_rol_actual, ['superadmin', 'admin_empresa'])) {
    redireccionar('/dashboard');
}

$id_usuario = intval($_GET['id'] ?? 0);
if ($id_usuario <= 0) {
    redireccionar('/usuarios');
}

$error = '';
$exito = '';

// Obtener usuario a editar
$query = "SELECT * FROM usuarios WHERE id = ?";
$usuario_edit = obtener_fila($query, 'i', [$id_usuario]);
if (!$usuario_edit) {
    redireccionar('/usuarios');
}

// Validar acceso para editar (admin_empresa solo su empresa)
if ($usuario_rol_actual === 'admin_empresa' && $usuario_edit['empresa_id'] != $empresa_id_actual) {
    redireccionar('/usuarios');
}

// Empresas para selección
$empresas = [];
if ($usuario_rol_actual === 'superadmin') {
    $empresas = obtener_filas("SELECT id, nombre FROM empresas WHERE activa=1 ORDER BY nombre");
} elseif ($usuario_rol_actual === 'admin_empresa') {
    $query_emp = "SELECT id, nombre FROM empresas WHERE id = ?";
    $empresas = obtener_filas($query_emp, 'i', [$empresa_id_actual]);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrf = $_POST['csrf_token'] ?? '';
    if (!verificar_csrf_token($csrf)) {
        $error = "Token de seguridad inválido";
    } else {
        $nombre = limpiar_entrada($_POST['nombre'] ?? '');
        $email = limpiar_entrada($_POST['email'] ?? '');
        $rol = limpiar_entrada($_POST['rol'] ?? '');
        $empresa_sel = intval($_POST['empresa_id'] ?? 0);
        $activo = isset($_POST['activo']) ? 1 : 0;
        
        $reset_password = trim($_POST['reset_password'] ?? '');
        $reset_password_conf = trim($_POST['reset_password_confirm'] ?? '');
        
        // Validar datos
        if (empty($nombre) || empty($email) || empty($rol) || empty($empresa_sel)) {
            $error = "Todos los campos obligatorios deben ser completados.";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = "Email no válido";
        } elseif ($rol === 'superadmin' && $usuario_rol_actual !== 'superadmin') {
            $error = "No tiene permisos para asignar rol de Super Administrador";
        } elseif ($usuario_rol_actual === 'admin_empresa' && $empresa_sel != $empresa_id_actual) {
            $error = "No tiene permisos para cambiar la empresa a otra";
        } elseif ($reset_password !== '' || $reset_password_conf !== '') {
            if ($reset_password !== $reset_password_conf) {
                $error = "Las contraseñas de reseteo no coinciden";
            } elseif (strlen($reset_password) < 6) {
                $error = "La nueva contraseña debe tener al menos 6 caracteres";
            }
        }
        
        if (!$error) {
            // Verificar email único (excepto este usuario)
            $existe = obtener_fila("SELECT id FROM usuarios WHERE email = ? AND id != ?", 'si', [$email, $id_usuario]);
            if ($existe) {
                $error = "El email ya está registrado por otro usuario";
            } else {
                $db = obtener_conexion();
                $db->begin_transaction();
                try {
                    $query_update = "UPDATE usuarios 
                                     SET nombre = ?, email = ?, rol = ?, empresa_id = ?, activo = ?
                                     WHERE id = ?";
                    ejecutar_query($query_update, 'sssiii', [$nombre, $email, $rol, $empresa_sel, $activo, $id_usuario]);
                    
                    // Resetear contraseña si corresponde
                    if ($reset_password !== '') {
                        $hash = hash_password($reset_password);
                        $query_pass = "UPDATE usuarios SET password_hash = ? WHERE id = ?";
                        ejecutar_query($query_pass, 'si', [$hash, $id_usuario]);
                    }
                    
                    registrar_log_seguridad("Usuario editado: $email", $_SESSION['usuario_id']);
                    $db->commit();
                    
                    set_flash("Usuario actualizado correctamente", 'success');
                    redireccionar('/usuarios');
                    
                } catch (Exception $e) {
                    $db->rollback();
                    $error = "Error actualizando usuario: " . $e->getMessage();
                    error_log("Error editando usuario: " . $e->getMessage());
                }
            }
        }
    }
}

$csrf_token = generar_csrf_token();

$checked_activo = $usuario_edit['activo'] == 1 ? 'checked' : '';

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Editar Usuario - <?php echo APP_NAME; ?></title>
    <style>
        /* Reutiliza CSS del módulo usuarios para consistencia */
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif; background: #f5f7fa; color: #333; }
        .navbar { background: white; border-bottom: 1px solid #e1e4e8; padding: 15px 30px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 2px 4px rgb(0 0 0 / 0.05); }
        .navbar-brand { font-size: 18px; font-weight: 600; color: #333; text-decoration: none; }
        .navbar-user { display: flex; align-items: center; gap: 20px; }
        .btn-logout { padding: 8px 16px; background: #dc3545; color: white; border-radius: 4px; font-size: 14px; text-decoration: none; }
        .container { max-width: 600px; margin: 30px auto; padding: 0 30px; }
        .page-header { margin-bottom: 30px; }
        .page-header h1 { font-size: 28px; }
        .alert-success { background: #efe; color: #3c3; border: 1px solid #cfc; padding: 10px 15px; border-radius: 6px; margin-bottom: 20px; }
        .alert-error { background: #fee; color: #c33; border: 1px solid #fcc; padding: 10px 15px; border-radius: 6px; margin-bottom: 20px; }
        form { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05); }
        label { display: block; margin-bottom: 6px; font-weight: 600; }
        input[type=text], input[type=email], input[type=password], select {
            width: 100%; padding: 10px 12px; margin-bottom: 18px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; font-family: inherit;
        }
        button { background: #667eea; color: white; border: none; padding: 12px 20px; border-radius: 6px; cursor: pointer; font-size: 16px; font-weight: 600; }
        button:hover { background: #5568d3; }
        a.back-link { display: inline-block; margin-bottom: 25px; color: #667eea; text-decoration: none; font-size: 14px; }
        a.back-link:hover { text-decoration: underline; }
        .checkbox-group { display: flex; align-items: center; gap: 10px; margin-bottom: 16px; }
        .checkbox-group input { width: auto; }
    </style>
</head>
<body>
    <nav class="navbar">
        <a href="/dashboard" class="navbar-brand"><?php echo APP_NAME; ?></a>
        <div class="navbar-user">
            <span><strong><?php echo htmlspecialchars($_SESSION['usuario_nombre']); ?></strong> (<?php echo ROLES[$_SESSION['usuario_rol']]; ?>)</span>
            <a href="/logout" class="btn-logout">Cerrar Sesión</a>
        </div>
    </nav>
    
    <div class="container">
        <a href="/usuarios" class="back-link">← Volver a Lista de Usuarios</a>
        
        <h1>Editar Usuario</h1>
        
        <?php if ($error): ?>
            <div class="alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
            
            <label for="nombre">Nombre completo</label>
            <input type="text" id="nombre" name="nombre" required maxlength="255" value="<?php echo htmlspecialchars($_POST['nombre'] ?? $usuario_edit['nombre']); ?>">
            
            <label for="email">Correo electrónico</label>
            <input type="email" id="email" name="email" required maxlength="255" value="<?php echo htmlspecialchars($_POST['email'] ?? $usuario_edit['email']); ?>">
            
            <label for="rol">Rol</label>
            <select id="rol" name="rol" required>
                <option value="">Seleccione rol</option>
                <?php foreach (['superadmin', 'admin_empresa', 'directorio', 'ejecutivo', 'supervisor', 'cliente'] as $k): ?>
                    <?php if ($k === 'superadmin' && $usuario_rol_actual !== 'superadmin') continue; ?>
                    <option value="<?php echo $k; ?>" <?php echo ((isset($_POST['rol']) && $_POST['rol'] == $k) || (!isset($_POST['rol']) && $usuario_edit['rol'] == $k)) ? 'selected' : ''; ?>>
                        <?php echo ROLES[$k]; ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <label for="empresa_id">Empresa</label>
            <select id="empresa_id" name="empresa_id" required>
                <option value="">Seleccione empresa</option>
                <?php foreach ($empresas as $e): ?>
                    <option value="<?php echo $e['id']; ?>" <?php echo ((isset($_POST['empresa_id']) && $_POST['empresa_id'] == $e['id']) || (!isset($_POST['empresa_id']) && $usuario_edit['empresa_id'] == $e['id'])) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($e['nombre']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            
            <div class="checkbox-group">
                <input type="checkbox" id="activo" name="activo" <?php echo $checked_activo; ?>>
                <label for="activo">Activo</label>
            </div>
            
            <h3>Opcional: Cambiar contraseña</h3>
            <label for="reset_password">Nueva contraseña</label>
            <input type="password" id="reset_password" name="reset_password" minlength="6">
            
            <label for="reset_password_confirm">Confirmar nueva contraseña</label>
            <input type="password" id="reset_password_confirm" name="reset_password_confirm" minlength="6">
            
            <button type="submit">Guardar Cambios</button>
        </form>
    </div>
</body>
</html>
