<?php
/**
 * Listado de Clientes
 * Versión: 2.0 Definitiva
 */

if (!defined('COMITEH_APP')) {
    die('Acceso no autorizado');
}

if (!verificar_sesion()) {
    redireccionar('/login');
}

// Solo usuarios con rol admin_empresa o superadmin pueden acceder
$usuario_id = $_SESSION['usuario_id'];
$usuario_rol = $_SESSION['usuario_rol'];
$empresa_id = $_SESSION['empresa_id'];

if (!in_array($usuario_rol, ['admin_empresa', 'superadmin'])) {
    redireccionar('/dashboard');
}

// Filtros
$search = trim($_GET['search'] ?? '');
$empresa_filtro = $_GET['empresa_id'] ?? '';
$estado_filtro = $_GET['estado'] ?? '';
$pagina = intval($_GET['pagina'] ?? 1);
$por_pagina = 20;
$offset = ($pagina - 1) * $por_pagina;

// Clausulas WHERE y parámetros para consulta
$where = ['1=1'];
$params = [];
$types = '';

// Filtros de acceso
if ($usuario_rol === 'admin_empresa') {
    $where[] = 'c.empresa_id = ?';
    $params[] = $empresa_id;
    $types .= 'i';
} elseif ($empresa_filtro && $empresa_filtro !== 'todas') {
    $where[] = 'c.empresa_id = ?';
    $params[] = intval($empresa_filtro);
    $types .= 'i';
}

// Filtro activo/inactivo
if ($estado_filtro !== '') {
    $activo_val = ($estado_filtro == '1') ? 1 : 0;
    $where[] = 'c.activo = ?';
    $params[] = $activo_val;
    $types .= 'i';
}

// Búsqueda
if ($search) {
    $where[] = '(c.nombre LIKE ? OR c.rut LIKE ? OR c.email LIKE ? OR c.telefono LIKE ?)';
    $param = '%' . $search . '%';
    $params[] = $param;
    $params[] = $param;
    $params[] = $param;
    $params[] = $param;
    $types .= 'ssss';
}

$where_sql = implode(' AND ', $where);

// Contar total clientes
$query_count = "SELECT COUNT(*) as total FROM clientes c WHERE $where_sql";
$total_row = obtener_fila($query_count, $types, $params);
$total_clientes = $total_row['total'] ?? 0;
$total_paginas = ceil($total_clientes / $por_pagina);

// Obtener clientes con paginación
$query = "SELECT c.*, e.nombre as empresa_nombre FROM clientes c
          JOIN empresas e ON c.empresa_id = e.id
          WHERE $where_sql
          ORDER BY c.nombre
          LIMIT ? OFFSET ?";

$params_pag = array_merge($params, [$por_pagina, $offset]);
$types_pag = $types . 'ii';

$clientes = obtener_filas($query, $types_pag, $params_pag);

// Para seleccionar empresas en filtro
$empresas = [];
if ($usuario_rol === 'superadmin') {
    $empresas = obtener_filas("SELECT id, nombre FROM empresas WHERE activa = 1 ORDER BY nombre");
}

// Mensaje flash
$flash = get_flash();

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Gestión de Clientes - <?php echo APP_NAME; ?></title>
    <style>
        /* CSS básico consistente con otros módulos */
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif; background: #f5f7fa; color: #333; }
        .navbar { background: white; border-bottom: 1px solid #e1e4e8; padding: 15px 30px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 2px 4px rgb(0 0 0 / 0.05); }
        .navbar-brand { font-size: 18px; font-weight: 600; color: #333; text-decoration: none; }
        .navbar-user { display: flex; align-items: center; gap: 20px; }
        .btn-logout { padding: 8px 16px; background: #dc3545; color: white; border-radius: 4px; font-size: 14px; text-decoration: none; }
        .container { max-width: 1400px; margin: 30px auto; padding: 0 30px; }
        .page-header { margin-bottom: 30px; display: flex; justify-content: space-between; align-items: center; }
        .page-header h1 { font-size: 28px; }
        .btn { padding: 10px 20px; border-radius: 6px; text-decoration: none; font-size: 14px; font-weight: 500; border: none; cursor: pointer; transition: all 0.3s; }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #5568d3; }
        .btn-secondary { background: #6c757d; color: white; }
        .btn-secondary:hover { background: #565e64; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-danger:hover { background: #c82333; }
        .filters { display: grid; grid-template-columns: repeat(auto-fit, minmax(170px, 1fr)); gap: 15px; margin-bottom: 20px; }
        .filter-group label { display: block; margin-bottom: 5px; font-size: 13px; font-weight: 500; color: #666; }
        .filter-group select, .filter-group input { width: 100%; padding: 8px 11px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px; }
        .filter-actions { display: flex; gap: 10px; align-items: flex-end; }
        .stats-row { display: flex; gap: 15px; margin-bottom: 20px; font-size: 14px; color: #666; }
        .card { background: white; border-radius: 8px; box-shadow: 0 2px 4px rgb(0 0 0 / 0.05); padding: 25px; margin-bottom: 20px; }
        table { width: 100%; border-collapse: collapse; }
        table th { background: #f8f9fa; padding: 12px; text-align: left; font-size: 13px; font-weight: 600; color: #666; border-bottom: 2px solid #e1e4e8; }
        table td { padding: 11px 12px; border-bottom: 1px solid #e1e4e8; font-size: 14px; }
        table tr:hover { background: #f8f9fa; cursor: pointer; }
        .badge-activo { background: #e8f5e9; color: #388e3c; }
        .badge-inactivo { background: #fbe9e7; color: #d84315; }
        .pagination { display: flex; justify-content: center; gap: 5px; margin-top: 20px; }
        .pagination a, .pagination span { padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; text-decoration: none; color: #333; }
        .pagination .active { background: #667eea; color: white; border-color: #667eea; }
        .pagination a:hover { background: #f8f9fa; }
        .alert-success { background: #efe; color: #3c3; border: 1px solid #cfc; padding: 10px 15px; border-radius: 6px; margin-bottom: 20px; }
        .alert-error { background: #fee; color: #c33; border: 1px solid #fcc; padding: 10px 15px; border-radius: 6px; margin-bottom: 20px; }
    </style>
</head>
<body>
    <nav class="navbar">
        <a href="/dashboard" class="navbar-brand"><?php echo APP_NAME; ?></a>
        <div class="navbar-user">
            <span><strong><?php echo htmlspecialchars($_SESSION['usuario_nombre']); ?></strong> (<?php echo ROLES[$_SESSION['usuario_rol']]; ?>)</span>
            <a href="/logout" class="btn-logout">Cerrar Sesión</a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <h1>Gestión de Clientes</h1>
            <a href="/clientes/crear" class="btn btn-primary">+ Nuevo Cliente</a>
        </div>
        
        <?php if ($flash): ?>
            <div class="alert-success"><?php echo htmlspecialchars($flash['mensaje']); ?></div>
        <?php endif; ?>

        <div class="card">
            <form method="GET" action="/clientes">
                <div class="filters">
                    <div class="filter-group">
                        <label>Búsqueda</label>
                        <input type="text" name="search" placeholder="Nombre, RUT, Email, Teléfono..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <?php if ($usuario_rol === 'superadmin'): ?>
                    <div class="filter-group">
                        <label>Empresa</label>
                        <select name="empresa_id">
                            <option value="todas">Todas</option>
                            <?php foreach ($empresas as $emp): ?>
                            <option value="<?php echo $emp['id']; ?>" <?php echo ($empresa_filtro == $emp['id'])?"selected":""; ?>>
                                <?php echo htmlspecialchars($emp['nombre']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <?php endif; ?>
                    <div class="filter-group">
                        <label>Estado</label>
                        <select name="estado">
                            <option value="">Todos</option>
                            <option value="1" <?php echo ($estado_filtro === '1')?"selected":""; ?>>Activo</option>
                            <option value="0" <?php echo ($estado_filtro === '0')?"selected":""; ?>>Inactivo</option>
                        </select>
                    </div>
                    <div class="filter-actions">
                        <button type="submit" class="btn btn-primary">Filtrar</button>
                        <a href="/clientes" class="btn btn-secondary">Limpiar</a>
                    </div>
                </div>
            </form>

            <div class="stats-row">
                Total de clientes: <?php echo number_format($total_clientes); ?> |
                Página: <?php echo $pagina; ?> de <?php echo max(1, $total_paginas); ?>
            </div>

            <?php if (empty($clientes)): ?>
                <p style="padding: 40px; text-align: center; color: #999;">No se encontraron clientes.</p>
            <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Nombre</th>
                        <th>RUT</th>
                        <th>Email</th>
                        <th>Teléfono</th>
                        <th>Empresa</th>
                        <th>Estado</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($clientes as $cliente): ?>
                        <tr onclick="window.location='/clientes/editar/<?php echo $cliente['id']; ?>'">
                            <td><?php echo htmlspecialchars($cliente['nombre']); ?></td>
                            <td><?php echo htmlspecialchars($cliente['rut']); ?></td>
                            <td><?php echo htmlspecialchars($cliente['email']); ?></td>
                            <td><?php echo htmlspecialchars($cliente['telefono']); ?></td>
                            <td><?php echo htmlspecialchars($cliente['empresa_nombre']); ?></td>
                            <td>
                                <?php if ($cliente['activo']): ?>
                                    <span class="badge-activo">Activo</span>
                                <?php else: ?>
                                    <span class="badge-inactivo">Inactivo</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="/clientes/editar/<?php echo $cliente['id']; ?>" class="btn btn-primary btn-small">Editar</a>
                                <?php if ($cliente['activo']): ?>
                                    <a href="/clientes/eliminar/<?php echo $cliente['id']; ?>" class="btn btn-danger btn-small"
                                        onclick="return confirm('¿Está seguro que desea desactivar este cliente?')">Desactivar</a>
                                <?php else: ?>
                                    <a href="/clientes/eliminar/<?php echo $cliente['id']; ?>" class="btn btn-secondary btn-small"
                                        onclick="return confirm('¿Está seguro que desea reactivar este cliente?')">Reactivar</a>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <?php if ($total_paginas > 1): ?>
                <div class="pagination">
                <?php for ($i = max(1,$pagina-2); $i <= min($total_paginas,$pagina+2); $i++): ?>
                    <?php if ($i==$pagina): ?>
                        <span class="active"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?pagina=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&empresa_id=<?php echo $empresa_filtro; ?>&estado=<?php echo $estado_filtro; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                </div>
            <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
