<?php
/**
 * Crear Nuevo Caso
 * Versión: 2.0 Definitiva
 */

if (!defined('COMITEH_APP')) {
    die('Acceso no autorizado');
}

if (!verificar_sesion()) {
    redireccionar('/login');
}

$usuario_id = $_SESSION['usuario_id'];
$usuario_nombre = $_SESSION['usuario_nombre'];
$usuario_rol = $_SESSION['usuario_rol'];
$empresa_id = $_SESSION['empresa_id'];

$error = '';
$exito = '';

// Obtener empresas (solo superadmin)
$empresas = [];
if ($usuario_rol === 'superadmin') {
    $query_empresas = "SELECT id, nombre, siglas FROM empresas WHERE activa = 1 ORDER BY nombre";
    $empresas = obtener_filas($query_empresas);
} elseif ($empresa_id) {
    $query_empresa = "SELECT id, nombre, siglas FROM empresas WHERE id = ? AND activa = 1";
    $empresas = obtener_filas($query_empresa, 'i', [$empresa_id]);
}

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrf_token = $_POST['csrf_token'] ?? '';
    
    if (!verificar_csrf_token($csrf_token)) {
        $error = 'Token de seguridad inválido';
    } else {
        // Obtener datos del formulario
        $form_empresa_id = intval($_POST['empresa_id'] ?? 0);
        $tipo_caso = limpiar_entrada($_POST['tipo_caso'] ?? '');
        $prioridad = limpiar_entrada($_POST['prioridad'] ?? 'media');
        $asunto = limpiar_entrada($_POST['asunto'] ?? '');
        $descripcion = limpiar_entrada($_POST['descripcion'] ?? '');
        
        // Datos del cliente
        $cliente_rut = limpiar_entrada($_POST['cliente_rut'] ?? '');
        $cliente_nombre = limpiar_entrada($_POST['cliente_nombre'] ?? '');
        $cliente_email = limpiar_entrada($_POST['cliente_email'] ?? '');
        $cliente_telefono = limpiar_entrada($_POST['cliente_telefono'] ?? '');
        
        // Validaciones
        if (empty($form_empresa_id)) {
            $error = 'Debe seleccionar una empresa';
        } elseif (empty($tipo_caso)) {
            $error = 'Debe seleccionar el tipo de caso';
        } elseif (empty($asunto)) {
            $error = 'El asunto es obligatorio';
        } elseif (empty($descripcion)) {
            $error = 'La descripción es obligatoria';
        } elseif (empty($cliente_rut)) {
            $error = 'El RUT del cliente es obligatorio';
        } elseif (!validar_rut($cliente_rut)) {
            $error = 'El RUT del cliente no es válido';
        } elseif (empty($cliente_nombre)) {
            $error = 'El nombre del cliente es obligatorio';
        } else {
            // Verificar que la empresa existe y el usuario tiene acceso
            if ($usuario_rol !== 'superadmin' && $form_empresa_id != $empresa_id) {
                $error = 'No tiene permiso para crear casos en esta empresa';
            } else {
                $db = obtener_conexion();
                $db->begin_transaction();
                
                try {
                    // Buscar o crear cliente
                    $query_cliente = "SELECT id FROM clientes WHERE empresa_id = ? AND rut = ? LIMIT 1";
                    $cliente_existente = obtener_fila($query_cliente, 'is', [$form_empresa_id, $cliente_rut]);
                    
                    if ($cliente_existente) {
                        $cliente_id = $cliente_existente['id'];
                        
                        // Actualizar datos del cliente
                        $query_update_cliente = "UPDATE clientes SET nombre = ?, email = ?, telefono = ? WHERE id = ?";
                        ejecutar_query($query_update_cliente, 'sssi', [$cliente_nombre, $cliente_email, $cliente_telefono, $cliente_id]);
                    } else {
                        // Crear nuevo cliente
                        $query_insert_cliente = "INSERT INTO clientes (empresa_id, rut, nombre, email, telefono, metodo_activacion, activo, consentimiento_datos, fecha_consentimiento) 
                                                VALUES (?, ?, ?, ?, ?, 'manual', 1, 1, NOW())";
                        $cliente_id = insertar($query_insert_cliente, 'issss', [$form_empresa_id, $cliente_rut, $cliente_nombre, $cliente_email, $cliente_telefono]);
                    }
                    
                    if (!$cliente_id) {
                        throw new Exception('Error al crear/actualizar cliente');
                    }
                    
                    // Generar número de caso
                    $numero_caso = generar_numero_caso($form_empresa_id, $tipo_caso);
                    if (!$numero_caso) {
                        throw new Exception('Error al generar número de caso');
                    }
                    
                    // Crear caso
                    $query_caso = "INSERT INTO casos (empresa_id, numero_caso, cliente_id, usuario_creador_id, tipo_caso, asunto, descripcion, estado, prioridad, fecha_creacion) 
                                   VALUES (?, ?, ?, ?, ?, ?, ?, 'nuevo', ?, NOW())";
                    $caso_id = insertar($query_caso, 'isiissss', [
                        $form_empresa_id,
                        $numero_caso,
                        $cliente_id,
                        $usuario_id,
                        $tipo_caso,
                        $asunto,
                        $descripcion,
                        $prioridad
                    ]);
                    
                    if (!$caso_id) {
                        throw new Exception('Error al crear el caso');
                    }
                    
                    // Registrar en historial
                    registrar_historial($caso_id, $usuario_id, 'Caso creado', 'Caso creado por ' . $usuario_nombre);
                    
                    // Registrar log de seguridad
                    registrar_log_seguridad('Caso creado: ' . $numero_caso, $usuario_id, ['caso_id' => $caso_id]);
                    
                    $db->commit();
                    
                    // Redireccionar al detalle del caso
                    redireccionar('/casos/detalle/' . $caso_id);
                    
                } catch (Exception $e) {
                    $db->rollback();
                    $error = 'Error al crear el caso: ' . $e->getMessage();
                    error_log("Error creando caso: " . $e->getMessage());
                }
            }
        }
    }
}

$csrf_token = generar_csrf_token();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Crear Nuevo Caso - <?php echo APP_NAME; ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #f5f7fa; color: #333; }
        .navbar { background: white; border-bottom: 1px solid #e1e4e8; padding: 15px 30px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 2px 4px rgba(0,0,0,0.05); }
        .navbar-brand { font-size: 18px; font-weight: 600; color: #333; text-decoration: none; }
        .navbar-user { display: flex; align-items: center; gap: 20px; }
        .btn-logout { padding: 8px 16px; background: #dc3545; color: white; text-decoration: none; border-radius: 4px; font-size: 14px; }
        .container { max-width: 1000px; margin: 30px auto; padding: 0 30px; }
        .page-header { margin-bottom: 30px; }
        .page-header h1 { font-size: 28px; margin-bottom: 10px; }
        .back-link { color: #667eea; text-decoration: none; font-size: 14px; }
        .back-link:hover { text-decoration: underline; }
        .card { background: white; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05); padding: 30px; }
        .alert { padding: 12px 15px; border-radius: 6px; margin-bottom: 20px; font-size: 14px; }
        .alert-error { background: #fee; color: #c33; border: 1px solid #fcc; }
        .alert-success { background: #efe; color: #3c3; border: 1px solid #cfc; }
        .form-section { margin-bottom: 30px; }
        .form-section h3 { font-size: 18px; margin-bottom: 15px; padding-bottom: 10px; border-bottom: 2px solid #f0f0f0; }
        .form-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 20px; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; margin-bottom: 8px; font-size: 14px; font-weight: 500; color: #333; }
        .form-group label .required { color: #dc3545; }
        .form-group input, .form-group select, .form-group textarea { width: 100%; padding: 10px 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; font-family: inherit; }
        .form-group textarea { min-height: 120px; resize: vertical; }
        .form-group input:focus, .form-group select:focus, .form-group textarea:focus { outline: none; border-color: #667eea; }
        .form-actions { display: flex; gap: 10px; justify-content: flex-end; margin-top: 30px; padding-top: 20px; border-top: 1px solid #f0f0f0; }
        .btn { padding: 12px 24px; border-radius: 6px; text-decoration: none; font-size: 14px; font-weight: 500; border: none; cursor: pointer; transition: all 0.3s; }
        .btn-primary { background: #667eea; color: white; }
        .btn-primary:hover { background: #5568d3; }
        .btn-secondary { background: #6c757d; color: white; }
        .btn-secondary:hover { background: #5a6268; }
        .help-text { font-size: 12px; color: #999; margin-top: 5px; }
    </style>
</head>
<body>
    <nav class="navbar">
        <a href="/dashboard" class="navbar-brand"><?php echo APP_NAME; ?></a>
        <div class="navbar-user">
            <span><strong><?php echo htmlspecialchars($usuario_nombre); ?></strong> (<?php echo ROLES[$usuario_rol]; ?>)</span>
            <a href="/logout" class="btn-logout">Cerrar Sesión</a>
        </div>
    </nav>
    
    <div class="container">
        <div class="page-header">
            <a href="/casos" class="back-link">← Volver a Gestión de Casos</a>
            <h1>Crear Nuevo Caso</h1>
        </div>
        
        <div class="card">
            <?php if ($error): ?>
                <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <?php if ($exito): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($exito); ?></div>
            <?php endif; ?>
            
            <form method="POST" action="">
                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                
                <!-- Sección: Información General -->
                <div class="form-section">
                    <h3>Información General del Caso</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Empresa <span class="required">*</span></label>
                            <select name="empresa_id" required>
                                <option value="">Seleccione empresa</option>
                                <?php foreach ($empresas as $emp): ?>
                                    <option value="<?php echo $emp['id']; ?>"><?php echo htmlspecialchars($emp['nombre']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Tipo de Caso <span class="required">*</span></label>
                            <select name="tipo_caso" required>
                                <option value="">Seleccione tipo</option>
                                <option value="reclamo">Reclamo</option>
                                <option value="consulta">Consulta</option>
                                <option value="sugerencia">Sugerencia</option>
                                <option value="contacto">Contacto</option>
                                <option value="felicitacion">Felicitación</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Prioridad <span class="required">*</span></label>
                            <select name="prioridad" required>
                                <option value="media" selected>Media</option>
                                <option value="baja">Baja</option>
                                <option value="alta">Alta</option>
                                <option value="urgente">Urgente</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Asunto <span class="required">*</span></label>
                        <input type="text" name="asunto" required maxlength="255" 
                               placeholder="Resumen breve del caso">
                    </div>
                    
                    <div class="form-group">
                        <label>Descripción Detallada <span class="required">*</span></label>
                        <textarea name="descripcion" required 
                                  placeholder="Describa en detalle el caso, problema o solicitud..."></textarea>
                    </div>
                </div>
                
                <!-- Sección: Datos del Cliente -->
                <div class="form-section">
                    <h3>Datos del Cliente</h3>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>RUT <span class="required">*</span></label>
                            <input type="text" name="cliente_rut" required 
                                   placeholder="12.345.678-9" maxlength="12">
                            <div class="help-text">Formato: 12.345.678-9</div>
                        </div>
                        
                        <div class="form-group">
                            <label>Nombre Completo <span class="required">*</span></label>
                            <input type="text" name="cliente_nombre" required 
                                   placeholder="Nombre completo del cliente">
                        </div>
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Email</label>
                            <input type="email" name="cliente_email" 
                                   placeholder="cliente@ejemplo.com">
                        </div>
                        
                        <div class="form-group">
                            <label>Teléfono</label>
                            <input type="text" name="cliente_telefono" 
                                   placeholder="+56 9 1234 5678" maxlength="20">
                        </div>
                    </div>
                </div>
                
                <div class="form-actions">
                    <a href="/casos" class="btn btn-secondary">Cancelar</a>
                    <button type="submit" class="btn btn-primary">Crear Caso</button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>
