<?php
/**
 * Funciones de Seguridad
 * Versión: 2.0 Definitiva
 */

if (!defined('COMITEH_APP')) {
    die('Acceso no autorizado');
}

/**
 * Iniciar sesión segura
 */
function iniciar_sesion() {
    if (session_status() === PHP_SESSION_NONE) {
        ini_set('session.cookie_httponly', 1);
        ini_set('session.cookie_secure', 1);
        ini_set('session.use_only_cookies', 1);
        
        session_name(SESSION_NAME);
        session_start();
        
        // Regenerar ID cada 5 minutos
        if (!isset($_SESSION['ultima_regeneracion'])) {
            $_SESSION['ultima_regeneracion'] = time();
        } elseif (time() - $_SESSION['ultima_regeneracion'] > 300) {
            session_regenerate_id(true);
            $_SESSION['ultima_regeneracion'] = time();
        }
    }
}

/**
 * Verificar sesión activa
 * @return bool
 */
function verificar_sesion() {
    iniciar_sesion();
    
    if (!isset($_SESSION['usuario_id']) || !isset($_SESSION['usuario_email'])) {
        return false;
    }
    
    // Verificar timeout
    $lifetime = ($_SESSION['usuario_rol'] === 'cliente') ? SESSION_LIFETIME_CLIENT : SESSION_LIFETIME;
    
    if (isset($_SESSION['ultima_actividad']) && (time() - $_SESSION['ultima_actividad'] > $lifetime)) {
        cerrar_sesion();
        return false;
    }
    
    $_SESSION['ultima_actividad'] = time();
    return true;
}

/**
 * Verificar rol de usuario
 * @param array $roles_permitidos
 * @return bool
 */
function verificar_rol($roles_permitidos = []) {
    if (!verificar_sesion()) {
        return false;
    }
    
    if (empty($roles_permitidos)) {
        return true;
    }
    
    return in_array($_SESSION['usuario_rol'], $roles_permitidos);
}

/**
 * Cerrar sesión
 */
function cerrar_sesion() {
    iniciar_sesion();
    $_SESSION = [];
    
    if (isset($_COOKIE[SESSION_NAME])) {
        setcookie(SESSION_NAME, '', time() - 3600, '/');
    }
    
    session_destroy();
}

/**
 * Generar token CSRF
 * @return string
 */
function generar_csrf_token() {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * Verificar token CSRF
 * @param string $token
 * @return bool
 */
function verificar_csrf_token($token) {
    return isset($_SESSION[CSRF_TOKEN_NAME]) && hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * Limpiar entrada
 * @param string $data
 * @return string
 */
function limpiar_entrada($data) {
    if (is_array($data)) {
        return array_map('limpiar_entrada', $data);
    }
    
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

/**
 * Validar RUT chileno
 * @param string $rut
 * @return bool
 */
function validar_rut($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', $rut);
    
    if (strlen($rut) < 2) return false;
    
    $dv = strtoupper(substr($rut, -1));
    $numero = substr($rut, 0, -1);
    
    $suma = 0;
    $multiplo = 2;
    
    for ($i = strlen($numero) - 1; $i >= 0; $i--) {
        $suma += $numero[$i] * $multiplo;
        $multiplo = ($multiplo == 7) ? 2 : $multiplo + 1;
    }
    
    $resto = $suma % 11;
    $dv_calculado = 11 - $resto;
    
    if ($dv_calculado == 11) $dv_calculado = '0';
    if ($dv_calculado == 10) $dv_calculado = 'K';
    
    return (string)$dv_calculado === $dv;
}

/**
 * Hash de contraseña
 * @param string $password
 * @return string
 */
function hash_password($password) {
    return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
}

/**
 * Verificar contraseña
 * @param string $password
 * @param string $hash
 * @return bool
 */
function verificar_password($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Registrar log de seguridad
 * @param string $evento
 * @param int $usuario_id
 * @param array $detalles
 */
function registrar_log_seguridad($evento, $usuario_id = null, $detalles = []) {
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    $query = "INSERT INTO logs_sistema (nivel, modulo, mensaje, usuario_id, ip_usuario, detalles) 
              VALUES (?, ?, ?, ?, ?, ?)";
    
    $detalles_json = json_encode($detalles);
    
    insertar($query, 'sssiss', [
        'warning',
        'seguridad',
        $evento,
        $usuario_id,
        $ip,
        $detalles_json
    ]);
}

/**
 * Redireccionar
 * @param string $ruta
 */
function redireccionar($ruta) {
    header("Location: " . APP_URL . "/" . ltrim($ruta, '/'));
    exit;
}

/**
 * Responder JSON
 * @param mixed $data
 * @param int $status
 */
function responder_json($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}
/**
 * Registrar un mensaje flash en sesión
 */
function set_flash($mensaje, $tipo = 'success') {
    $_SESSION['flash'] = ['mensaje' => $mensaje, 'tipo' => $tipo];
}

/**
 * Obtener mensaje flash y borrarlo
 */
function get_flash() {
    if (!empty($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}
